#!/bin/bash
set -ex

usage() {
    echo "basename ${0} get all src from Gitlab"
    echo "--help | -h                      Print this help"
    echo "--token-file= | -t <file>        Set file with Gitlab token"
    echo "--gitlab-url=* | -g <url>        Set Gtilab URL"
    exit 1
}

# Map long options to short ones and rebuild argv
translated=()
while (( $# )); do
    case "$1" in
        --help)          translated+=("-h"); shift ;;
        --token-file=*)  translated+=("-t" "${1#*=}"); shift ;;
        --gitlab-url=*)  translated+=("-g" "${1#*=}"); shift ;;
        --)              translated+=("--"); shift; translated+=("$@"); set -- ;;
        *)               translated+=("$1"); shift ;;
      esac
done
set -- "${translated[@]}"

declare GITLAB_URL="https://git.mws-team.ru"
declare TOKEN_FILE=""

while getopts ":ht:g:" opt; do
    case "$opt" in
        h) usage ;;
        t) TOKEN_FILE=$OPTARG ;;
        g) GITLAB_URL=$OPTARG ;;
        :) echo "Missing arg for -$OPTARG"; usage ;;
        \?) echo "Invalid option: -$OPTARG"; usage ;;
    esac
done

function checks {
    if [ -z "${TOKEN_FILE}" ]
    then
        echo "Missing token file"
        usage
    fi
    
    if [ -z "${GITLAB_URL}" ]
    then
        echo "Missing gitlab url"
        usage
    fi
    
    if [ ! -f "${TOKEN_FILE}" ]
    then
        echo "Token file not found"
        usage
    fi
}

function get_subgroups {
    for PAGE in $(seq 1 $(curl -X HEAD -w '%{header_json}' --header "Authorization: Bearer ${TOKEN}" ${GITLAB_URL}/api/v4/groups/${1}/subgroups?per_page=100 2>/dev/null| jq -r '."x-total-pages"[0]'))
    do
        curl --silent --header "Authorization: Bearer ${TOKEN}" "${GITLAB_URL}/api/v4/groups/${1}/subgroups?per_page=100&page=${PAGE}" | jq -r '.[].id'
    done
}

function clone_project {
    PROJECT_INFO=$(curl --silent --header "Authorization: Bearer ${TOKEN}" ${GITLAB_URL}/api/v4/projects/${1} | jq .)
    echo "INFO: Check project $(echo ${PROJECT_INFO} | jq -r '.path_with_namespace')"
    if [ ! -d "$(echo ${PROJECT_INFO} | jq -r '.path_with_namespace')" ]
    then
        echo "INFO: Clone project $(echo ${PROJECT_INFO} | jq -r '.path_with_namespace')"
        mkdir -p $(echo ${PROJECT_INFO} | jq -r '.path_with_namespace')
        git clone $(echo ${PROJECT_INFO} | jq -r '. | "\(.ssh_url_to_repo) \(.path_with_namespace)"')
    fi
}

checks

TOKEN=$(cat ${TOKEN_FILE})

for PAGE in $(seq 1 $(curl -X HEAD -w '%{header_json}' --header "Authorization: Bearer ${TOKEN}" ${GITLAB_URL}/api/v4/groups?per_page=100 2>/dev/null| jq -r '."x-total-pages"[0]'))
do
    for GROUP in $(curl --silent --header "Authorization: Bearer ${TOKEN}" "${GITLAB_URL}/api/v4/groups?per_page=100&page=${PAGE}" | jq -r '.[].id')
    do
        for P_PAGE in $(seq 1 $(curl -X HEAD -w '%{header_json}' --header "Authorization: Bearer ${TOKEN}"  ${GITLAB_URL}/api/v4/groups/${GROUP}/projects?per_page=100 2>/dev/null| jq -r '."x-total-pages"[0]'))
        do
            for PROJECT in $(curl --silent --header "Authorization: Bearer ${TOKEN}" "${GITLAB_URL}/api/v4/groups/${GROUP}/projects?per_page=100&page=${P_PAGE}" | jq -r '.[].id')
            do
                clone_project ${PROJECT}
            done
        done
        for SG_PAGE in $(seq 1 $(curl -X HEAD -w '%{header_json}' --header "Authorization: Bearer ${TOKEN}"  ${GITLAB_URL}/api/v4/groups/${GROUP}/subgroups?per_page=100 2>/dev/null | jq -r '."x-total-pages"[0]'))
        do
            for SUBGROUP in $(curl --silent --header "Authorization: Bearer ${TOKEN}" "${GITLAB_URL}/api/v4/groups/${GROUP}/subgroups?per_page=100&page=${SG_PAGE}" | jq -r '.[].id')
            do
                for P_SG_PAGE in $(seq 1 $(curl -X HEAD -w '%{header_json}' --header "Authorization: Bearer ${TOKEN}"  ${GITLAB_URL}/api/v4/groups/${SUBGROUP}/projects?per_page=100 2>/dev/null| jq -r '."x-total-pages"[0]'))
                do
                    for PROJECT in $(curl --silent --header "Authorization: Bearer ${TOKEN}" "${GITLAB_URL}/api/v4/groups/${SUBGROUP}/projects?per_page=100&page=${P_SG_PAGE}" | jq -r '.[].id')
                    do
                        clone_project ${PROJECT}
                    done
                done
            done
        done
    done
done
echo "OK"
